import { supabase } from '@/config/supabase';
import {
  Sketch,
  AIStatus,
  EnhancementType,
  UploadSketchRequest,
  EnhanceSketchRequest
} from './types';
// import { AIService } from '@/modules/ai/service'; // Assume AIService.enhanceSketch exists

export const sketchesService = {
  // Upload sketch (file + metadata)
  async uploadSketch(vendor: any, file: any, fields: UploadSketchRequest) {
    // Enforce feature gate and plan limits externally
    // 1. Upload file to Supabase Storage
    const fileExt = file.originalname.split('.').pop();
    const filePath = `${vendor.id}/${Date.now()}_${file.originalname}`;
    const { data: storageData, error: storageError } = await supabase.storage.from('style-references').upload(filePath, file.buffer, {
      contentType: file.mimetype
    });
    if (storageError) return { error: storageError };
    const file_url = supabase.storage.from('style-references').getPublicUrl(filePath).publicUrl;
    // 2. Insert sketch record
    const { data, error } = await supabase.from('sketches').insert({
      vendor_id: vendor.id,
      customer_id: fields.customer_id || null,
      invoice_id: fields.invoice_id || null,
      file_url,
      ai_status: 'none',
      tags: fields.tags || [],
      notes: fields.notes || null,
      uploaded_at: new Date().toISOString()
    }).select().single();
    return { data, error };
  },

  // Request AI enhancement
  async enhanceSketch(vendor: any, sketch_id: string, req: EnhanceSketchRequest) {
    // 1. Check ownership
    const { data: sketch, error: fetchError } = await supabase.from('sketches').select('*').eq('id', sketch_id).eq('vendor_id', vendor.id).single();
    if (fetchError || !sketch) return { error: fetchError || { message: 'Sketch not found' } };
    // 2. Update ai_status to processing
    await supabase.from('sketches').update({ ai_status: 'processing', enhancement_type: req.enhancement_type }).eq('id', sketch_id);
    // 3. Trigger AI enhancement (async, placeholder)
    // await AIService.enhanceSketch(sketch_id, req.enhancement_type);
    // 4. Return updated sketch
    const { data: updated } = await supabase.from('sketches').select('*').eq('id', sketch_id).single();
    return { data: updated };
  },

  // List sketches (with filters)
  async listSketches(vendor_id: string, filters: any) {
    let query = supabase.from('sketches').select('*').eq('vendor_id', vendor_id);
    if (filters.customer_id) query = query.eq('customer_id', filters.customer_id);
    if (filters.invoice_id) query = query.eq('invoice_id', filters.invoice_id);
    if (filters.tags) query = query.contains('tags', filters.tags);
    const { data, error } = await query.order('uploaded_at', { ascending: false });
    return { data, error };
  },

  // Get single sketch
  async getSketch(vendor_id: string, id: string) {
    const { data, error } = await supabase.from('sketches').select('*').eq('id', id).eq('vendor_id', vendor_id).single();
    return { data, error };
  },

  // Delete sketch
  async deleteSketch(vendor_id: string, id: string) {
    // Only allow if not linked to finalized invoice
    const { data: sketch, error: fetchError } = await supabase.from('sketches').select('invoice_id').eq('id', id).eq('vendor_id', vendor_id).single();
    if (fetchError || !sketch) return { error: fetchError || { message: 'Sketch not found' } };
    if (sketch.invoice_id) {
      // Check if invoice is finalized
      const { data: invoice } = await supabase.from('invoices').select('status').eq('id', sketch.invoice_id).single();
      if (invoice && invoice.status === 'finalized') {
        return { error: { message: 'Cannot delete sketch linked to finalized invoice.' } };
      }
    }
    const { data, error } = await supabase.from('sketches').delete().eq('id', id).eq('vendor_id', vendor_id).select().single();
    return { data, error };
  }
}; 